﻿// This script manages the timing and flow of the test simulator. It is also responsible for telling the UI when and how to update

using System.Collections;
using UnityEngine;
using UnityEngine.SceneManagement;
using UnityStandardAssets.Vehicles.Car;
using UnityStandardAssets.Utility;


public class GameManager : MonoBehaviour
{
	// The game manager holds a public static reference to itself. This is often referred to as being a "singleton" and allows it to be access from all other objects in the scene.
	// This should be used carefully and is generally reserved for "manager" type objects
	public static GameManager instance;		

	[Header("Test Settings")]
	public int numberOfLaps = 3;			// the number of laps to complete
	public CarController vehicleMovement;	// a reference to the car's VehicleMovement script
	public GameObject car;					// a reference to the car's VehicleMovement script

	[Header("UI References")]
	public CarUI carUI;						// a reference to the car's CarUI script
	public LapTimeUI lapTimeUI;				// a reference to the LapTimeUI script in the scene
	public GameObject testComplete;         // a reference to the UI canvas that appears when the test is complete
	public CarAIControl carAIControl;		// a reference to the car's AI control script
	public TestStats testStats;				// a reference to the test stats script

	float[] lapTimes;						// an array containing the car's lap times
	int currentLap = 0;						// the current lap the test is on
	bool isTestOver;						// a flag to determine if the test is over
	bool testHasBegun;                      // a flag to determine if the test has begun
    private float lapStartTime = 0f;		// a variable to start the time at zero


    [Header("Car Starting Position")]
    public Transform carStartPosition;		// an empty GameObject to help position the car
	public WaypointProgressTracker waypointProgressTracker; // a reference for the car's path around the scene
	private int restartLap = 0;
    


    void Awake()
	{
		// if the variable instance has not be initialized, set it equal to this GameManager script...
		if (instance == null)
			instance = this;
		//...Otherwise, if there already is a GameManager and it isn't this, destroy this
		//(there can only be one GameManager)
		else if (instance != this)
			Destroy(gameObject);


	}

	void OnEnable()
	{
		// when the GameManager is enabled, we start a coroutine to handle the setup of
		// the game. It is done this way to allow our intro cutscene to work. By slightly
		// delaying the start of the test, we give the cutscene time to take control and 
		// play out
		StartCoroutine(Init());
	}

    IEnumerator Init()
    {
        // wait a little while to let everything initialize
        yield return new WaitForSeconds(.1f);

        // initialize the lapTimes array and set that the race has begun
        lapTimes = new float[numberOfLaps];
        testHasBegun = true;

        // set the lapStartTime to the current time
        lapStartTime = Time.time;
    }



    void Update()
	{
		// update the speed on the cars UI
		UpdateUI_Speed();

		// update the steering angle on the cars UI
		UpdateUI_CarSteerAngle();

        // if we have an active game...
        if (IsActiveGame())
        {
            // ...calculate the time for the lap and update the UI
            lapTimes[currentLap] = Time.time - lapStartTime;
            UpdateUI_LapTime();
        }

		// if the car's rigidbody is activated, call method
        if (testHasBegun && vehicleMovement.m_Rigidbody )
		{
			CarCrashed();
		}
    }

	// called by the FinishLine script
	public void PlayerCompletedLap()
	{
		// if the test is already over exit this method 
		if (isTestOver)
			return;

		// incremement the current lap
		currentLap++;

		// if the player has completed the required amount of laps...
		if (currentLap >= numberOfLaps)
		{
			//...the test is now over...
			isTestOver = true;

			//call stats
			testStats.DispayStats();

			//...update the laptime UI...
			UpdateUI_FinalTime();

			//...and show the Test Stats UI
			testComplete.SetActive(true);

			// stop car
			carAIControl.m_Driving = false;
		}
        else
        {
            // update the lap start time for the next lap
            lapStartTime = Time.time;
        }
    }


    void UpdateUI_LapTime()
	{
		// if we have a LapTimeUI reference, update it
		if (lapTimeUI != null)
			lapTimeUI.SetLapTime(currentLap, lapTimes[currentLap]);
	}

	void UpdateUI_FinalTime()
	{
		// if we have a LapTimeUI reference... 
		if (lapTimeUI != null)
		{
			float total = 0f;

			//...loop through all of the lapTimes and total up an amount...
			for (int i = 0; i < lapTimes.Length; i++)
				total += lapTimes[i];

			//... and update the final test time
			lapTimeUI.SetFinalTime(total);
		}
	}

    // method to update the steering angle on the cars UI
    void UpdateUI_CarSteerAngle()
	{
		carUI.SetCarSteerAngle();
	}

    // if we have a VehicleMovement and ShipUI reference, update it
    void UpdateUI_Speed()
	{
		if (vehicleMovement != null && carUI != null) 
			carUI.SetSpeedDisplay (Mathf.Abs(vehicleMovement.mphSpeed));
	}

	// if the race has begun and the game is not over, we have an active game
	public bool IsActiveGame()
	{
		return testHasBegun && !isTestOver;
	}

	// if the car crashes and is moving very slowly when the test is not over...
	public void CarCrashed()
	{
        if (vehicleMovement.mphSpeed < 0.01f && !isTestOver)
        {
            // restart the current lap
            restartLap = currentLap;
            RestartAfterCrash();
        }
    }

	// restart the car at the beginning of the test and reset its waypoint
    public void RestartAfterCrash()
    {
        // reset the car's position and rotation
        car.transform.position = carStartPosition.position;
        car.transform.rotation = carStartPosition.rotation;

        // reset the car's speed and angular velocity
        vehicleMovement.m_Rigidbody.linearVelocity = Vector3.zero;
        vehicleMovement.m_Rigidbody.angularVelocity = Vector3.zero;

		// reset the next waypoint the car is looking for
        waypointProgressTracker.Reset();

        // update the lap start time
        lapStartTime = Time.time;
    }

	// restart the test by reloading the scene that is currently loaded
    public void Restart()
    {
        SceneManager.LoadScene(SceneManager.GetActiveScene().name);
    }

    // when the restart button is clicked, delay the event by 1 second
    public void RestartClicked()
	{
		Invoke ("Restart", 1f);
	}
}